<?php
/**
 * Print class
 *
 * @package woocommerce-print-invoice-delivery-notes
 */

/**
 * Exit if accessed directly
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Print class
 */
if ( ! class_exists( 'ar_invoice_Print' ) ) {

	/**
	 * WooCommerce Delivery Notes Print class.
	 */
	class ar_invoice_Print {

		/**
		 * Template registrations
		 *
		 * @var array $template_registrations
		 */
		public static $template_registrations;

		/**
		 * Template styles
		 *
		 * @var array $template_styles
		 */
		public static $template_styles;

		/**
		 * Default Template
		 *
		 * @var array $template
		 */
		public $template;

		/**
		 * Query vars
		 *
		 * @var array $query_vars
		 */
		public $query_vars;

		/**
		 * Order IDs
		 *
		 * @var array $order_ids
		 */
		public $order_ids;

		/**
		 * Order email
		 *
		 * @var array $order_email
		 */
		public $order_email;

		/**
		 * Orders
		 *
		 * @var array $orders
		 */
		public $orders;

		/**
		 * Constructor
		 */
		public function __construct() {
			// Define the templates.
			self::$template_registrations = apply_filters(
				'ar_invoice_template_registration',
				array(
					apply_filters(
						'ar_invoice_template_registration_invoice',
						array(
							'type'   => 'invoice',
							'labels' => array(
								'name'           => __( 'Invoice', THEME_NAME ),
								'name_plural'    => __( 'Invoices', THEME_NAME ),
								'print'          => __( 'Print Invoice', THEME_NAME ),
								'print_plural'   => __( 'Print Invoices', THEME_NAME ),
								'message'        => __( 'Invoice created.', THEME_NAME ),
								'message_plural' => __( 'Invoices created.', THEME_NAME ),
								'setting'        => __( 'Show "Print Invoice" button', THEME_NAME ),
							),
						)
					),
					

				)
			);

			// Add the default template as first item after filter hooks passed.
			array_unshift(
				self::$template_registrations,
				array(
					'type'   => 'order',
					'labels' => array(
						'name'           => __( 'Order', THEME_NAME ),
						'name_plural'    => __( 'Orders', THEME_NAME ),
						'print'          => __( 'Print Order', THEME_NAME ),
						'print_plural'   => __( 'Print Orders', THEME_NAME ),
						'message'        => null,
						'message_plural' => null,
						'setting'        => null,
					),
				)
			);

			// Default template.
			$this->template = self::$template_registrations[0];


			// Insert the query vars.
			$this->query_vars = array(
				'print-order-type',
				'print-order-email',
			);

			// Load the hooks.
			add_action( 'init', array( $this, 'load_hooks' ) );
			add_filter( 'query_vars', array( $this, 'add_query_vars' ) );
			add_action( 'parse_request', array( $this, 'parse_request' ) );
			add_action( 'template_redirect', array( $this, 'template_redirect_theme' ) );
			add_action( 'wp_ajax_print_order', array( $this, 'template_redirect_admin' ) );
		}

		/**
		 * Load the init hooks
		 */
		public function load_hooks() {
			add_rewrite_endpoint( 'print-order', EP_PAGES );

			// Flush the rules when the transient is set.
			// This is important to make the endpoint work.
			if ( '1' === get_transient( 'ar_invoice_flush_rewrite_rules' ) ) {
				delete_transient( 'ar_invoice_flush_rewrite_rules' );
				flush_rewrite_rules();
			}
		}

		/**
		 * Add the query vars to the url
		 *
		 * @param array $vars Query variables.
		 */
		public function add_query_vars( $vars ) {
			foreach ( $this->query_vars as $var ) {
				$vars[] = $var;
			}
			return $vars;
		}

		/**
		 * Parse the query variables
		 *
		 * @param object $wp WP Object.
		 */
		public function parse_request( $wp ) {
			if ( isset( $_GET['print-order' ] ) ) {
				// changed.
				$wdn_get_end_point_var  = sanitize_text_field( wp_unslash( $_GET['print-order' ] ) );
				$wp->query_vars['print-order' ] = $wdn_get_end_point_var;
			} elseif ( isset( $wp->query_vars['print-order' ] ) ) {
				$wp->query_vars['print-order' ] = $wp->query_vars['print-order' ];
			}
		}

		/**
		 * Template handling in the front-end
		 */
		public function template_redirect_theme() {
			global $wp;
			// Check the page url and display the template when on my-account page.
			if ( ! empty( $wp->query_vars['print-order'] ) && is_account_page() ) {
				$type  = ! empty( $wp->query_vars['print-order-type'] ) ? $wp->query_vars['print-order-type'] : null;
				$email = ! empty( $wp->query_vars['print-order-email'] ) ? $wp->query_vars['print-order-email'] : null;
				$this->generate_template( $wp->query_vars['print-order'], $type, $email );
				exit;
			}
		}

		/**
		 * Template handling in the back-end
		 */
		public function template_redirect_admin() {
			// Let the backend only access the page.
			// changed.
			if ( is_admin() && current_user_can( 'edit_shop_orders' ) && ! empty( $_REQUEST['print-order'] ) && ! empty( $_REQUEST['action'] ) ) {
				$type  = ! empty( $_REQUEST['print-order-type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['print-order-type'] ) ) : null;
				$email = ! empty( $_REQUEST['print-order-email'] ) ? sanitize_email( wp_unslash( $_REQUEST['print-order-email'] ) ) : null;
				// changed.
				$wdn_get_print_order = isset( $_GET['print-order'] ) ? sanitize_text_field( wp_unslash( $_GET['print-order'] ) ) : '';
				$this->generate_template( $wdn_get_print_order, $type, $email );
				exit;
			}
			exit;
		}

		/**
		 * Generate the template
		 *
		 * @param array  $order_ids Order IDs.
		 * @param string $template_type Template type.
		 * @param string $order_email  Order email.
		 */
		public function generate_template( $order_ids, $template_type = 'order', $order_email = null ) {
			global $post, $wp;

			// Explode the ids when needed.
			if ( ! is_array( $order_ids ) ) {
				$this->order_ids = array_filter( explode( '-', $order_ids ) );
			}

			// Set the current template.
			foreach ( self::$template_registrations as $template_registration ) {
				if ( $template_type === $template_registration['type'] ) {
					$this->template = $template_registration;
					break;
				}
			}

			// Set the email.
			if ( empty( $order_email ) ) {
				$this->order_email = null;
			} else {
				$this->order_email = strtolower( $order_email );
			}

			// Create the orders and check permissions.
			$populated = $this->populate_orders();

			// Only continue if the orders are populated.
			if ( ! $populated ) {
				die();
			}

			// Load the print template html.
			$location = get_template_directory() . '/include/ar_invoices/templates/';
			$args     = array();
			wc_get_template( 'print-order.php', $args, $location, $location );
			exit;
		}


		/**
		 * Get print page url
		 *
		 * @param array   $order_ids Order ids.
		 * @param string  $template_type Template type.
		 * @param string  $order_email Order email.
		 * @param boolean $permalink Permalink.
		 */
		public function get_print_page_url( $order_ids, $template_type = 'order', $order_email = null, $permalink = false ) {
			// Explode the ids when needed.
			if ( ! is_array( $order_ids ) ) {
				$order_ids = array_filter( explode( '-', $order_ids ) );
			}

			// Build the args.
			$args = array();

			// Set the template type arg.
			foreach ( self::$template_registrations as $template_registration ) {
				if ( $template_type === $template_registration['type'] && 'order' !== $template_type ) {
					$args = wp_parse_args( array( 'print-order-type' => $template_type ), $args );
					break;
				}
			}

			// Set the email arg.
			if ( ! empty( $order_email ) ) {
				$args = wp_parse_args( array( 'print-order-email' => $order_email ), $args );
			}

			// Generate the url.
			$order_ids_slug = implode( '-', $order_ids );
			$endpoint = 'print-order';

			// Create another url depending on where the user prints. This prevents some issues with ssl when the my-account page is secured with ssl but the admin isn't.
			if ( is_admin() && current_user_can( 'edit_shop_orders' ) && false === $permalink ) {
				// For the admin we use the ajax.php for better security.
				$args     = wp_parse_args( array( 'action' => 'print_order' ), $args );
				$base_url = admin_url( 'admin-ajax.php' );

				// Add the order ids and create the url.
				$url = add_query_arg( $endpoint, $order_ids_slug, $base_url );
			} else {
				// For the theme.
				$base_url = wc_get_page_permalink( 'myaccount' );
				

				// Add the order ids and create the url.
				if ( get_option( 'permalink_structure' ) ) {
					$url = trailingslashit( trailingslashit( $base_url ) . $endpoint . '/' . $order_ids_slug );
				} else {
					$url = add_query_arg( $endpoint, $order_ids_slug, $base_url );
				}
			}

			// Add all other args.
			$url = add_query_arg( $args, $url );

			return esc_url( $url );
		}

		/**
		 * Create the orders list and check the permissions.
		 */
		private function populate_orders() {
			$this->orders = array();

			// Get the orders.
			$posts = wc_get_orders(
				array(
					'limit'       => -1,
					'orderby'     => 'date',
					'order'       => 'DESC',
					'post__in'    => $this->order_ids,
					'post_status' => 'any',
				)
			);

			// All orders should exist.
			if ( count( $posts ) !== count( $this->order_ids ) ) {
				$this->orders = null;
				return false;
			}

			// Check permissons of the user to determine if the orders should be populated.
			foreach ( $posts as $post ) {
				$order = $post;

				$wdn_order_id = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_id() : $order->id;
				// Logged in users.
				if ( is_user_logged_in() && ( ! current_user_can( 'edit_shop_orders' ) && ! current_user_can( 'view_order', $wdn_order_id ) ) ) {
					$this->orders = null;
					return false;
				}

				$wdn_order_billing_id = ( version_compare( get_option( 'woocommerce_version' ), '3.0.0', '>=' ) ) ? $order->get_billing_email() : $order->billing_email;

				// An email is required for not logged in users.
				if ( ! is_user_logged_in() && ( empty( $this->order_email ) || strtolower( $wdn_order_billing_id ) !== $this->order_email ) ) {
					$this->orders = null;
					return false;
				}

				// Save the order to get it without an additional database call.
				$this->orders[ $wdn_order_id ] = $order;
			}
			return true;
		}

		/**
		 * Get the order.
		 *
		 * @param int $order_id Order id.
		 */
		public function get_order( $order_id ) {
			if ( isset( $this->orders[ $order_id ] ) ) {
				return $this->orders[ $order_id ];
			}
			return false;
		}


		/**
		 * Get the order invoice date.
		 *
		 * @param int $order_id Order id.
		 */
		public function get_order_invoice_date( $order_id ) {
			$order = wc_get_order( $order_id );

			// Get the invoice date.
			$meta_date = $order->get_meta( '_ar_invoice_invoice_date', true );
			if ( '' === $meta_date ) {
				$order->add_meta_data( '_ar_invoice_invoice_date', time(), true );
				$order->save();
			}

			$formatted_date = date_i18n( get_option( 'date_format' ), $meta_date );
			return apply_filters( 'ar_invoice_order_invoice_date', $formatted_date, $meta_date );
		}

	}

}
?>
